//
//  YHIFamilyInfoListCell.swift
//  galaxy
//
//  Created by edy on 2024/8/17.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import AttributedString

enum YHFamilyInfoType: Int {
    case none = 0
    case mainApplicantName = 1
    case firstName = 2
    case lastName = 3
    case marriage = 4
    case birthNation = 5
    case birthPlaceInChina = 6
    case birthPlaceAboard = 7
    case declareNation = 8
    case occupation = 9
    case hongKongId = 10
    case travelCertificateType = 11
    case travelCertificateId = 12
    case childName = 13
    case spouseName = 14
}

enum YHFamilyInfoActionType: Int {
    case none = 0
    case edit = 1    // 输入框编辑
    case select = 2  // 弹出弹窗选择
    case choice = 3  // 按钮多项选择
}

class YHFamilyInfoOption {
    var value: String = ""
    var index: Int = -1
    var isSelect: Bool = false
    
    init(index: Int, value: String, isSelect: Bool) {
        self.index = index
        self.value = value
        self.isSelect = isSelect
    }
}

class YHFamilyInfoItem {
    var id: YHFamilyInfoType = .none
    var action: YHFamilyInfoActionType = .none
    var isMust: Bool = false
    var title: String = ""
    var value: String = ""
    var previousTitle: String = ""
    var previousValue: String = ""
    var options: [YHFamilyInfoOption] = []
    var isEditMode: Bool = false
    var isShowPreviousInfo: Bool = false
    
    var tips: String {
        get {
            return self.getRedTips()
        }
    }
    
    func getRedTips() -> String {

        if !self.value.isEmpty {  // value不为空
            if self.id == .hongKongId, !self.value.isHKIdentityCardNumber() {
                return "请输入正确的香港身份证号"
            }
            return ""
        }
        
        if self.isMust == false { // 非必填
            return ""
        }
        
        // 必填
        // value 为空的情况
        if self.id == .mainApplicantName {
            return "请输入主申人姓名"
        }
            
        if self.id == .spouseName {
            return "请输入配偶姓名"
        }
        
        if self.id == .childName {
            return "请输入子女姓名"
            
        } else if self.id == .firstName {
            return "请输入姓(拼音)"
            
        } else if self.id == .lastName {
            return "请输入名(拼音)"
            
        } else if self.id == .birthPlaceAboard {
            return "请输入出生地"

        } else if self.id == .occupation {
            return "请输入职业"
            
        } else if self.id == .hongKongId {
            return "请输入香港身份证号"
        }
        return ""
    }
    
    required init() {
        
    }
    
    init(id: YHFamilyInfoType, action: YHFamilyInfoActionType, isMust: Bool, title: String, value: String, previousTitle: String, previousValue: String, options: [YHFamilyInfoOption] = [], isEditMode: Bool, isShowPreviousInfo: Bool) {
        self.id = id
        self.action = action
        self.isMust = isMust
        self.title = title
        self.value = value
        self.previousTitle = previousTitle
        self.previousValue = previousValue
        self.isEditMode = isEditMode
        self.isShowPreviousInfo = isShowPreviousInfo
        self.options = options
    }
}

class YHIFamilyInfoListCell: UITableViewCell {

    static let cellReuseIdentifier = "YHIFamilyInfoListCell"
    var familyInfo: YHFamilyMemberInfoModel = YHFamilyMemberInfoModel()
    
    var isEditState: Bool = false {
        didSet {
            let text = self.isEditState ? "完成" : "修改"
            self.modifyBtn.setTitle(text, for: .normal)
        }
    }
    
    // 是否进入编辑模式
    var editBlock: ((Bool) -> Void)?
    // 选择出生国家
    var selectBirthNation: ((Bool) -> Void)?
    // 姓名更改
    var updateName: ((String) -> Void)?
    
    var items: [YHFamilyInfoItem] = []
    
    lazy var whiteView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        view.layer.cornerRadius = kCornerRadius6
        return view
    }()
    
    lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.numberOfLines = 0
        return label
    }()
    
    lazy var modifyBtn: UIButton = {
        let btn = UIButton()
        btn.titleLabel?.font = UIFont.PFSC_R(ofSize: 14)
        btn.setTitleColor(.brandMainColor, for: .normal) 
        btn.setTitle("修改", for: .normal)
        btn.addTarget(self, action: #selector(didModifyBtnClicked), for: .touchUpInside)
        btn.YH_clickEdgeInsets = UIEdgeInsets(top: 15, left: 15, bottom: 15, right: 15)
        return btn
    }()
    
    lazy var itemsContentView: UIView = {
        let view = UIView()
        return view
    }()
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    @objc func didModifyBtnClicked() {
        editBlock?(!self.isEditState)
    }
    
    func setupUI() {
        
        self.selectionStyle = .none
        self.backgroundColor = .clear
        self.contentView.backgroundColor = .clear
        contentView.addSubview(whiteView)
        whiteView.addSubview(titleLabel)
        whiteView.addSubview(modifyBtn)
        whiteView.addSubview(itemsContentView)
        
        whiteView.snp.makeConstraints { make in
            make.top.equalTo(16.0)
            make.left.equalTo(16)
            make.right.equalTo(-16)
            make.bottom.equalTo(0)
        }
        
        titleLabel.snp.makeConstraints { make in
            make.top.equalTo(18)
            make.left.equalTo(18)
            make.right.equalTo(-18)
        }
        
        modifyBtn.snp.makeConstraints { make in
            make.top.equalTo(18)
            make.right.equalTo(-18)
            make.size.equalTo(CGSize(width: 28, height: 20))
        }
        
        itemsContentView.snp.makeConstraints { make in
            make.top.equalTo(titleLabel.snp.bottom).offset(12)
            make.left.equalTo(18)
            make.right.equalTo(-18)
            make.bottom.equalTo(0)
        }

    }
    
    func updateItems(_ items: [YHFamilyInfoItem]) {
        self.items =  items
        itemsContentView.removeSubviews()
        
        var lastItemView: UIView?
        for (index, item) in items.enumerated() {
            var itemView: UIView? = YHInfoItemView(frame: .zero)
            if item.action == .choice {
                itemView = YHInfoOptionView(frame: .zero)
            }
            if let itemView = itemView as? YHInfoItemView {
                itemsContentView.addSubview(itemView)
                itemView.updateItem(item, familyInfo: familyInfo)
                itemView.updateName = {
                    [weak self] _ in
                    guard let self = self else { return }
                    
                    for subView in self.itemsContentView.subviews {
                        if subView is YHInfoItemView {
                            if let targetView = subView as? YHInfoItemView {
                                if targetView.item.id == .firstName {
                                    targetView.item.value = self.familyInfo.familyName
                                    targetView.updateItem(targetView.item, familyInfo: self.familyInfo)
                                }
                                if targetView.item.id == .lastName {
                                    targetView.item.value = self.familyInfo.givenName
                                    targetView.updateItem(targetView.item, familyInfo: self.familyInfo)
                                }
                            }
                        }
                    }
                }
                
            } else if let itemView = itemView as? YHInfoOptionView {
                itemView.clickBlock = {
                    [weak self] selectItem in
                    guard let self = self else { return }
                    if item.id == .birthNation {
                        familyInfo.birthPlaceAboard = "\(selectItem.index)"
                        self.selectBirthNation?(familyInfo.birthPlaceAboard != "1")
                    }
                }
                itemsContentView.addSubview(itemView)
                itemView.updateItem(item, familyInfo: familyInfo)
            }
            
            if let itemView = itemView {
                itemView.snp.makeConstraints { make in
                    make.left.right.equalTo(0)
                    if let lastView = lastItemView {
                        make.top.equalTo(lastView.snp.bottom)
                    } else {
                        make.top.equalTo(itemsContentView.snp.top)
                    }
                    
                    if index == items.count-1 {
                        make.bottom.equalTo(itemsContentView.snp.bottom)
                    }
                }
                lastItemView = itemView
            }
            
        }
        self.setNeedsLayout()
        self.layoutIfNeeded()
    }

}
